<?php
namespace App\Http\Controllers\Api;

use App\Events\RideAcceptedEvent;
use App\Http\Controllers\Controller;
use App\Models\BookingManagement;
use App\Models\DriverManagement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RideController extends Controller
{
    public function accept(Request $request)
    {
        $request->validate([
            'booking_id' => 'required|exists:booking_managements,id',
            'driver_id'  => 'required|exists:driver_managements,id',
        ]);

        $booking = BookingManagement::find($request->booking_id);

        if ($booking->driver_id != 0) {
            return response()->json(['message' => 'Ride already accepted.'], 403);
        }

        // Assign the booking to the driver
        $booking->driver_id = $request->driver_id;
        $booking->save();

        // Optionally update driver status
        DriverManagement::where('id', $request->driver_id)->update(['trip_status' => 1]);

        return response()->json(['message' => 'Ride accepted successfully.']);
    }

    public function acceptRide(Request $request)
    {
        $request->validate([
            'booking_id' => 'required|exists:booking_management,id',
            'driver_id'  => 'required|exists:driver_management,id',
            'accept'     => 'required|boolean',
        ]);

        $booking = BookingManagement::find($request->booking_id);
        $driver  = DriverManagement::find($request->driver_id);

        if ($booking->status !== 'pending') {
            return response()->json(['message' => 'Ride already handled'], 400);
        }

        if ($request->accept) {
            DB::transaction(function () use ($booking, $driver) {
                // Update booking
                $booking->update([
                    'driver_id' => $driver->id,
                    'status'    => 'accepted',
                ]);

                // Update driver status
                $driver->update(['trip_status' => 1]);
            });

            // Notify admin/customer
            event(new RideAcceptedEvent($booking, $driver));

            return response()->json(['message' => 'Ride accepted successfully']);
        }

        return response()->json(['message' => 'Ride rejected']);
    }
}
