<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class DriverStatusController extends Controller
{

    public function getDriverStatus($driver_id)
    {
        $today    = Carbon::today();
        $tomorrow = Carbon::tomorrow();

        // Get total online seconds
        $onlineSessions = DB::table('driver_log_times')
            ->where('driver_id', $driver_id)
            ->whereBetween('login_time', [$today, $tomorrow])
            ->whereNotNull('logout_time')
            ->get();

        $totalSeconds = 0;

        foreach ($onlineSessions as $session) {
            $login  = Carbon::parse($session->login_time);
            $logout = Carbon::parse($session->logout_time);
            $totalSeconds += $logout->diffInSeconds($login);
        }

        // Convert to hours:minutes
        $hours       = floor($totalSeconds / 3600);
        $minutes     = floor(($totalSeconds % 3600) / 60);
        $onlineHours = sprintf('%02d:%02d', $hours, $minutes);

        // Get existing driver + booking data
        $driverInfo = DB::table('driver_management as dm')
            ->leftJoin('booking_management as bm', 'dm.id', '=', 'bm.driver_id')
            ->leftJoin('driver_arrived as da', 'dm.id', '=', 'da.driver_id')
            ->where('dm.id', $driver_id)
            ->select(
                'dm.id as driver_id',
                'dm.driver_code',
                'dm.driver_name',
                'da.driver_status',
                'bm.id as booking_id',
                'bm.from_address',
                'bm.to_address',
                'bm.booking_status'
            )
            ->orderByDesc('bm.id')
            ->first();

        if (! $driverInfo) {
            return response()->json([
                'status'  => false,
                'message' => 'Driver or booking not found.',
            ], 404);
        }

        // Prepare response
        return response()->json([
            'status' => true,
            'data'   => [
                'driver'  => [
                    'driver_id'     => $driverInfo->driver_id,
                    'driver_code'   => $driverInfo->driver_code,
                    'driver_name'   => $driverInfo->driver_name,
                    'driver_status' => $driverInfo->driver_status,
                    'online_hours'  => $onlineHours, // ⏰ Added here
                ],
                'booking' => [
                    'booking_id'     => $driverInfo->booking_id,
                    'from_address'   => $driverInfo->from_address,
                    'to_address'     => $driverInfo->to_address,
                    'booking_status' => $driverInfo->booking_status,
                ],
            ],
        ]);
    }

}
