<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Services\ShirtDraftService;
use App\Services\PantDraftService;
use App\Services\BlouseDraftService;
use Illuminate\Http\Request;
use App\Models\Measurement;
use App\Models\Customer;


class MeasurementManagementController extends Controller
{

    public function index()
    {
        $measurements = Measurement::with('customer')->latest()->paginate(10);
        return view('admin.measurement_management.list', compact('measurements'));
    }

    public function create()
    {
        $customers = Customer::orderBy('name')->get();
        return view('admin.measurement_management.add', compact('customers'));
    }
 public function store(Request $request)
{
    $request->validate([
        'customer_id'  => 'required|exists:customers,id',
        'garment_type' => 'required|string',
        'unit'         => 'required|in:inch,cm',
        'measurement'  => 'required|array'
    ]);

    $garmentType = $request->garment_type;
    $unit = $request->unit;

    // Get measurement array for selected garment
    $measurementData = $request->measurement[$garmentType];

    /**
     * Convert CM → INCH (store inch internally)
     */
    if ($unit === 'cm') {
        $measurementData = $this->convertCmToInch($measurementData);
    }

    Measurement::create([
        'customer_id'     => $request->customer_id,
        'garment_type'    => $garmentType,
        'measurement_data'=> $measurementData, // JSON
        'unit'            => 'inch', // always store inch
        'notes'           => $request->notes,
    ]);

    return redirect()->route('admin.MeasurementManagement-index')
        ->with('success', 'Measurement added successfully');
}

  public function show($id)
{
    $measurement = Measurement::with('customer')->findOrFail($id);

    $draft = null;
    if ($measurement->garment_type === 'Shirt') {
        $draft = ShirtDraftService::calculate($measurement);
    }
     if ($measurement->garment_type === 'Pant') {
        $draft = PantDraftService::calculate($measurement);
    }
    if ($measurement->garment_type === 'Blouse') {
        $draft = BlouseDraftService::calculate($measurement);
    }

    return view(
        'admin.measurement_management.view',
        compact('measurement', 'draft')
    );
}

    public function edit($id)
    {

        $customers = Customer::orderBy('name')->get();
       $measurement = Measurement::with('customer')->findOrFail($id);

        return view('admin.measurement_management.edit', compact('measurement', 'customers'));
    }

  public function update(Request $request, $id)
    {
    $measurement = Measurement::findOrFail($id);

    $request->validate([
        'garment_type' => 'required|string',
        'unit'         => 'required|in:inch,cm',
    ]);

    $unit = $request->unit;

    $chest    = $request->chest;
    $waist    = $request->waist;
    $hip      = $request->hip;
    $shoulder = $request->shoulder;
    $sleeve   = $request->sleeve;
    $length   = $request->length;

    // ✅ Convert CM → INCH
    if ($unit === 'cm') {
        $chest    = $chest / 2.54;
        $waist    = $waist / 2.54;
        $hip      = $hip / 2.54;
        $shoulder = $shoulder / 2.54;
        $sleeve   = $sleeve / 2.54;
        $length   = $length / 2.54;
    }

    $measurement->update([
        'garment_type'=> $request->garment_type,
        'unit'        => $unit,

        'chest'       => $chest,
        'waist'       => $waist,
        'hip'         => $hip,
        'shoulder'    => $shoulder,
        'sleeve'      => $sleeve,
        'length'      => $length,

        'notes'       => $request->notes,
    ]);

    return redirect()->route('admin.MeasurementManagement-index')
        ->with('success', 'Measurement updated successfully');
}


    public function destroy($id)
    {
        Measurement::findOrFail($id)->delete();

        return redirect()->route('admin.MeasurementManagement-index')
            ->with('success', 'Measurement deleted successfully');
    }
    private function convertCmToInch($data)
    {
    foreach ($data as $key => $value) {

        // If nested (Chudithar top/bottom)
        if (is_array($value)) {
            $data[$key] = $this->convertCmToInch($value);
        }
        // Convert only numeric values
        elseif (is_numeric($value)) {
            $data[$key] = round($value / 2.54, 2);
        }
    }

    return $data;
    }

}
